/*
 * Decompiled with CFR 0.152.
 */
package com.baidu.aip.speech;

import com.baidu.aip.speech.AipSpeech;
import com.baidu.aip.speech.BaiduWebSocketClient;
import com.baidu.aip.speech.BaiduWebSocketListener;
import com.baidu.aip.speech.StreamTtsListener;
import com.baidu.aip.speech.StreamTtsRequest;
import java.net.URI;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.java_websocket.exceptions.WebsocketNotConnectedException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StreamTtsSession {
    private static final Logger LOGGER = LoggerFactory.getLogger(StreamTtsSession.class);
    private static final int DEFAULT_START_TIMEOUT_MILLISECONDS = 60000;
    private static final int DEFAULT_STOP_TIMEOUT_MILLISECONDS = 60000;
    private final StreamTtsRequest request;
    private final StreamTtsListener listener;
    private BaiduWebSocketClient client;
    private AipSpeech speechClient;
    private CountDownLatch startLatch;
    private CountDownLatch stopLatch;
    private volatile boolean isStarted = false;
    private volatile boolean isStopped = false;
    private volatile boolean hasError = false;

    StreamTtsSession(AipSpeech speechClient, StreamTtsRequest request, StreamTtsListener listener) {
        this.speechClient = speechClient;
        this.request = request != null ? request : new StreamTtsRequest();
        this.listener = listener;
    }

    public void start() throws Exception {
        this.start(60000L);
    }

    public void start(long timeoutMillis) throws Exception {
        if (this.isStarted) {
            throw new IllegalStateException("Synthesizer already started");
        }
        if (this.listener == null) {
            throw new IllegalArgumentException("Listener cannot be null");
        }
        this.startLatch = new CountDownLatch(1);
        this.stopLatch = new CountDownLatch(1);
        String accessToken = this.speechClient.getAccessToken();
        if (accessToken == null) {
            throw new RuntimeException("Access token cannot be null!");
        }
        if (this.request.getPer() < 0) {
            throw new RuntimeException("Per must be a valid value!");
        }
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append("wss://aip.baidubce.com/ws/2.0/speech/publiccloudspeech/v1/tts");
        urlBuilder.append("?access_token=").append(accessToken);
        urlBuilder.append("&per=").append(this.request.getPer());
        URI uri = URI.create(urlBuilder.toString());
        LOGGER.debug("Try to Connect WebSocket URL: {}", (Object)urlBuilder.toString());
        this.client = new BaiduWebSocketClient(uri, new BaiduWebSocketListenerImpl());
        boolean connected = this.client.connectBlocking(timeoutMillis, TimeUnit.MILLISECONDS);
        if (!connected) {
            throw new Exception("WebSocket connection failed!");
        }
        LOGGER.debug("WebSocket Connect Success!");
        StartMessage startMessage = new StartMessage(this.request);
        this.client.send(startMessage.toJson());
        LOGGER.debug("Waiting for startup to complete, timeout period: {} ms", (Object)timeoutMillis);
        boolean result = this.startLatch.await(timeoutMillis, TimeUnit.MILLISECONDS);
        if (!result) {
            throw new Exception("Start timeout after " + timeoutMillis + " milliseconds");
        }
        this.isStarted = true;
    }

    public void process(String text) throws Exception {
        if (!this.isStarted || this.hasError) {
            throw new IllegalStateException("Synthesizer not started or has error");
        }
        if (this.isStopped) {
            throw new IllegalStateException("Synthesizer already stopped");
        }
        if (this.client == null || !this.client.isOpen()) {
            throw new WebsocketNotConnectedException();
        }
        TextMessage textMessage = new TextMessage(text);
        this.client.send(textMessage.toJson());
    }

    public void stop() throws Exception {
        this.stop(60000L);
    }

    public void stop(long timeoutMillis) throws Exception {
        if (!this.isStarted || this.hasError) {
            throw new IllegalStateException("Synthesizer not started or has error");
        }
        if (this.isStopped) {
            return;
        }
        if (this.client != null && this.client.isOpen()) {
            FinishMessage finishMessage = new FinishMessage();
            this.client.send(finishMessage.toJson());
            if (timeoutMillis > 0L) {
                LOGGER.debug("Waiting for completion to stop, timeout period: {} ms", (Object)timeoutMillis);
                boolean result = this.stopLatch.await(timeoutMillis, TimeUnit.MILLISECONDS);
                if (!result) {
                    throw new Exception("Stop timeout after " + timeoutMillis + " milliseconds");
                }
            } else {
                this.stopLatch.await();
            }
        }
        this.isStopped = true;
    }

    public void close() {
        try {
            if (this.client != null) {
                this.client.close();
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private static class FinishMessage {
        private FinishMessage() {
        }

        public String toJson() {
            JSONObject json = new JSONObject();
            json.put("type", (Object)"system.finish");
            return json.toString();
        }
    }

    private static class TextMessage {
        private final String text;

        public TextMessage(String text) {
            this.text = text;
        }

        public String toJson() {
            JSONObject json = new JSONObject();
            json.put("type", (Object)"text");
            JSONObject payload = new JSONObject();
            payload.put("text", (Object)this.text);
            json.put("payload", (Object)payload);
            return json.toString();
        }
    }

    private static class StartMessage {
        private final StreamTtsRequest request;

        public StartMessage(StreamTtsRequest request) {
            this.request = request;
        }

        public String toJson() {
            JSONObject json = new JSONObject();
            json.put("type", (Object)"system.start");
            JSONObject payload = new JSONObject();
            payload.put("spd", this.request.getSpd());
            payload.put("pit", this.request.getPit());
            payload.put("vol", this.request.getVol());
            payload.put("aue", this.request.getAue());
            if (this.request.getAudioCtrl() != null && !this.request.getAudioCtrl().isEmpty()) {
                payload.put("audio_ctrl", (Object)this.request.getAudioCtrl());
            }
            json.put("payload", (Object)payload);
            return json.toString();
        }
    }

    private class BaiduWebSocketListenerImpl
    implements BaiduWebSocketListener {
        private boolean synthesisStartedNotified = false;

        private BaiduWebSocketListenerImpl() {
        }

        @Override
        public void onOpen() {
            LOGGER.debug("WebSocket connection is open!");
        }

        @Override
        public void onMessage(String message) {
            LOGGER.debug("Message received: {}", (Object)message);
            try {
                JSONObject json = new JSONObject(message);
                String type = json.optString("type");
                String code = json.optString("code");
                String infoMessage = json.optString("message");
                if ("system.started".equals(type)) {
                    if ("0".equals(code)) {
                        if (!this.synthesisStartedNotified) {
                            this.notifySynthesisStart();
                        }
                    } else {
                        this.handleErrorResponse(type, code, infoMessage, true);
                    }
                } else if ("system.error".equals(type)) {
                    LOGGER.debug("Received system error: {}", (Object)message);
                    this.handleErrorResponse(type, code, infoMessage, false);
                } else if ("system.finished".equals(type)) {
                    if ("0".equals(code)) {
                        LOGGER.debug("Received system completion message!");
                        this.releaseLatches();
                    } else {
                        this.handleErrorResponse(type, code, infoMessage, true);
                    }
                } else {
                    LOGGER.debug("error message: {}", (Object)message);
                }
            }
            catch (Exception e) {
                StreamTtsSession.this.hasError = true;
                LOGGER.debug("Error parsing message: {}", (Object)e.getMessage(), (Object)e);
            }
        }

        @Override
        public void onMessage(byte[] data) {
            if (!this.synthesisStartedNotified) {
                this.notifySynthesisStart();
            }
            if (StreamTtsSession.this.listener != null) {
                StreamTtsSession.this.listener.onAudioData(StreamTtsSession.this.request.getSessionId(), data);
            }
        }

        @Override
        public void onClose(int code, String reason, boolean remote) {
            if (code != 1000) {
                LOGGER.error("WebSocket connection stopped, code: {}, reason: {}, remote: {}", new Object[]{code, reason, remote});
            }
            if (StreamTtsSession.this.listener != null) {
                StreamTtsSession.this.listener.onSynthesisEnd(StreamTtsSession.this.request.getSessionId());
            }
            this.releaseLatches();
        }

        @Override
        public void onError(Exception ex) {
            StreamTtsSession.this.hasError = true;
            LOGGER.debug("WebSocket error: {}", (Object)ex.getMessage(), (Object)ex);
            JSONObject errorResponse = new JSONObject();
            errorResponse.put("error_msg", (Object)ex.getMessage());
            errorResponse.put("session_id", (Object)StreamTtsSession.this.request.getSessionId());
            if (StreamTtsSession.this.listener != null) {
                StreamTtsSession.this.listener.onSynthesisFail(errorResponse);
            }
            this.releaseLatches();
        }

        private void notifySynthesisStart() {
            if (!this.synthesisStartedNotified) {
                this.synthesisStartedNotified = true;
                if (StreamTtsSession.this.listener != null) {
                    StreamTtsSession.this.listener.onSynthesisStart(StreamTtsSession.this.request.getSessionId());
                }
                if (StreamTtsSession.this.startLatch != null) {
                    StreamTtsSession.this.startLatch.countDown();
                }
            }
        }

        private void handleErrorResponse(String type, String code, String infoMessage, boolean setErrorFlag) {
            if (setErrorFlag) {
                StreamTtsSession.this.hasError = true;
            }
            String defaultMessage = "System error occurred";
            if ("system.started".equals(type)) {
                defaultMessage = "System start failed";
            } else if ("system.finished".equals(type)) {
                defaultMessage = "System finish failed";
            }
            JSONObject errorResponse = new JSONObject();
            errorResponse.put("error_msg", (Object)(infoMessage != null && !infoMessage.isEmpty() ? infoMessage : defaultMessage));
            errorResponse.put("session_id", (Object)StreamTtsSession.this.request.getSessionId());
            errorResponse.put("error_code", (Object)code);
            if (StreamTtsSession.this.listener != null) {
                StreamTtsSession.this.listener.onSynthesisFail(errorResponse);
            }
            this.releaseLatches();
        }

        private void releaseLatches() {
            if (StreamTtsSession.this.startLatch != null) {
                StreamTtsSession.this.startLatch.countDown();
            }
            if (StreamTtsSession.this.stopLatch != null) {
                StreamTtsSession.this.stopLatch.countDown();
            }
        }
    }
}

